"""
@Project: LinalgDat2022
@File: BasicExtensions.py

@Description: Project A basic extensions.

"""
import math
import sys

sys.path.append('../Core')
from Vector import Vector
from Matrix import Matrix

__author__ = "Magnus Goltermann, University of Copenhagen"
__date__ = "10/5/22"
__version__ = "0.0.1"


def AugmentRight(A: Matrix, v: Vector) -> Matrix:
    """
    Create an augmented matrix from a matrix A and a vector v.

    See page 12 in 'Linear Algebra for Engineers and Scientists'
    by K. Hardy.

    :param A: a matrix of size M-by-N.
    :param v: a column vector of size M.

    :return: a matrix of size M-by-(N + 1)
    """
    M = A.M_Rows
    N = A.N_Cols
    B = Matrix(M, N + 1)
    for i in range(M):
        for j in range(N):
            B[i, j] = A[i, j]
    for i in range(M):
        B[i, N] = v[i]

    return B

"""
Returnerer en vektor, som er et produkt af matricen A og vektoren v.
Dette gøres ved at iterere over koloner og rækker, og så lægge resultat (som i almindelig)
matrice multiplikation i y[i].
"""
def MatVecProduct(A: Matrix, v: Vector) -> Vector:
    """
    This function computes the matrix-vector product of a matrix A
    and a column vector v

    See page 68 in "Linear Algebra for Engineers and Scientists"
    by K. Hardy.
    :param A: an M-by-N Matrix.
    :param v: a size N Vector.

    :return: a size M Vector y such that y = A.v
    """
    M = A.M_Rows
    N = A.N_Cols
    y = Vector(M)
    for i in range(M):
        for j in range(N):
            y[i] = y[i]+A[i, j]*v[j]
    return y



"""
Returnerer en matrice, som er et produkt af matrice A og B.
Dette gøres ved at iterere over kolonner og rækker, og så lægge resultat (som i almindelig)
matrice multiplikation i product_matrix[i,j].
"""
def MatrixProduct(A: Matrix, B: Matrix) -> Matrix:
    """
    Compute the Matrix product of two given matrices A and B.

    See page 58 in "Linear Algebra for Engineers and Scientists"
    by K. Hardy.

    :param A: an M-by-N Matrix.
    :param B: an N-by-P Matrix.

    :returns: the M-by-P Matrix A*B.
    """
    M_A = A.M_Rows
    N_A = A.N_Cols
    P_B = B.N_Cols
    product_matrix = Matrix(M_A, P_B)
    for n in range(N_A):        
        for i in range(M_A):
            for j in range(P_B):
                product_matrix[i, j] = product_matrix[i, j] + A[i, n]*B[n, j]
    return product_matrix


"""
Returnerer matricen A transponeret.
Dette gøres ved at lave en ny matrice B som laves ved at iterere over kolononer og rækker,
og sætter rækkerne til kolonnerne og kolonnerne til rækkerne i matrice A over til matrice B.
"""
def Transpose(A: Matrix) -> Matrix:
    """
    Computes the transpose of a given Matrix.

    See page 69 in "Linear Algebra for Engineers and Scientists"
    by K. Hardy.

    :param A: A M-by-N Matrix.
    :returns: A N-by-M Matrix B such that B = A^T.
    """
    M = A.M_Rows
    N = A.N_Cols
    B = Matrix(N, M)
    for i in range(N):
        for j in range(M):
            B[i,j] = A[j,i]
    return B

"""
Giver længden af vektoren.
Dette gøres ved at bruge den euklidiske afstandsformel:
Itererer over hvert element i vektoren, tager elementet i anden og summer dem alle.
Herefter tages kvadratroden.
"""
def VectorNorm(v: Vector) -> float:
    """
    Computes the Euclidean Vector norm of a given Vector.

    See page 197 in "Linear Algebra for Engineers and Scientists"
    by K.Hardy.

    :param v: An N - dimensional Vector.
    :return: The Euclidean norm of the Vector.
    """
    n = int(v.size())
    norm = 0
    for i in range(n):
        norm = norm + v.__getitem__(i)**2
    return math.sqrt(norm)



